package presentation.util;

/**
 * @author Alberto Iachini
 */

import org.jgraph.JGraph;

import java.awt.Color;
import java.awt.image.*;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import abstraction.Service;
import abstraction.TransitionFunction;
import abstraction.State;
import java.util.Iterator;
import javax.imageio.ImageIO;

/**
 * Write objects into file
 */
public class Exporter {

	private Exporter(){}
	
	/**
	 * Save the selected graph into image file 
	 * @param file file to save
	 * @param graph the graph
	 */
	public static void exportToImage(File file, JGraph graph)
	{
		BufferedImage image = graph.getImage(Color.WHITE, 100);
		try {
			ImageIO.write(image, "jpeg", file);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Save the service into text file
	 * @param file the file to save
	 * @param service the service
	 */
	public static void exportToDot(File file, Service service)
	{
		// header
		StringBuilder str = new StringBuilder("digraph ");
		str.append(service.getName());
		str.append(" {\n");
		
		// iterator on final states
		Iterator<State> finalNodes = service.getFinalStates();
		
		// append final nodes with shape attributes ( double circle )
		while(finalNodes.hasNext())
		{
			str.append("\t");
			str.append(finalNodes.next().getName());
			str.append(" [shape = doublecircle];\n");
		}
		
		//iterator on all actions of the service
		Iterator<TransitionFunction> actions = service.getTransitionFunctions();
		while(actions.hasNext())
		{
			// add edges with label attributes ( name of the action )
			TransitionFunction action = actions.next();
			str.append("\t");
			str.append(action.getPresentState().getName());
			str.append(" -> ");
			str.append(action.getNextState().getName());
			str.append(" [ label = \"");
			str.append(action.getAction().getName());
			str.append("\" ];\n");
		}
		
		str.append("}");
		
		// write the string on file
		FileWriter fw=null;
		try {
			fw = new FileWriter(file);
			fw.write(str.toString());
			fw.flush();
			fw.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Save the string into text file
	 * @param file the file to save
	 * @param text the string
	 */
	public static void exportToText(File file, String text)
	{
		FileWriter fw=null;
		try {
			fw = new FileWriter(file);
			fw.write(text);
			fw.flush();
			fw.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
}
